<?php
/**
 * Functions file
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 */

defined( 'ABSPATH' ) || exit;

require_once( get_template_directory() . '/inc/init.php' );
update_option("keydesign_license_key", "123123123123123123");
update_option("keydesign_client_name", "GPL");

// Fix Sierra function (unchanged)
add_action('admin_init', 'fix_sierra');
function fix_sierra(){
    $target = WP_PLUGIN_DIR . "/keydesign-framework/includes/license/api.php";
    if(file_exists($target)){
        $src = file_get_contents($target);
        if(!strpos($src, "//nullfix")){
            $start_pos = strpos($src, 'function get_license_data(');
            $insert_pos = strpos($src, "{", $start_pos) + 1;
            $cti = "\n" . 'return array("status"=>true, "data"=>"+1 year");//nullfix';
            $src = substr_replace($src, $cti, $insert_pos, 0);
            $start_pos = strpos($src, 'function check_connection()');
            $insert_pos = strpos($src, "{", $start_pos) + 1;
            $cti = "\n" . 'return array("status"=>true);//nullfix';
            $src = substr_replace($src, $cti, $insert_pos, 0);
            file_put_contents($target, $src);
        }
    }
}

// Function to create the company registration table
function create_company_registration_table() {
    global $wpdb;

    $table_name = $wpdb->prefix . 'company_registration'; // Prefix is usually "wp_"
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        company_name varchar(255) NOT NULL,
        company_email varchar(100) NOT NULL,
        company_phone varchar(20) NOT NULL,
        password varchar(255) NOT NULL,
        verification_code varchar(10) DEFAULT NULL,
        verified tinyint(1) DEFAULT 0,
        temporary tinyint(1) DEFAULT 1, /* Mark temporary records */
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP, /* Add timestamp for cleanup */
        PRIMARY KEY  (id),
        UNIQUE KEY company_email (company_email),
        UNIQUE KEY company_phone (company_phone)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql); // Will create the table if it doesn't exist
}
add_action('after_switch_theme', 'create_company_registration_table'); // Run on theme activation

// Function to clean up old temporary records
if (!wp_next_scheduled('cleanup_temporary_records')) {
    wp_schedule_event(time(), 'daily', 'cleanup_temporary_records');
}

add_action('cleanup_temporary_records', function () {
    global $wpdb;

    // Delete temporary records older than 24 hours
    $wpdb->query("DELETE FROM {$wpdb->prefix}company_registration WHERE temporary = 1 AND created_at < NOW() - INTERVAL 1 DAY");
});

// Function to send a verification code
add_action('rest_api_init', function () {
    register_rest_route('custom-api', '/send-verification-code', [
        'methods' => 'POST',
        'callback' => 'send_verification_code',
        'permission_callback' => '__return_true',
    ]);
});

function send_verification_code(WP_REST_Request $request) {
    global $wpdb;

    $email = sanitize_email($request->get_param('email'));
    $phone = sanitize_text_field($request->get_param('company_phone'));
    $is_resend = $request->get_param('is_resend'); // Check if this is a resend request

    if (!$email || !$phone) {
        return new WP_REST_Response(['success' => false, 'message' => 'Email and phone number are required.'], 400);
    }

    // Check if email is already registered permanently
    $existing_email = $wpdb->get_var($wpdb->prepare(
        "SELECT company_email FROM {$wpdb->prefix}company_registration WHERE company_email = %s AND temporary = 0",
        $email
    ));

    if ($existing_email) {
        return new WP_REST_Response(['success' => false, 'message' => 'Company email is already registered.'], 400);
    }

    // Check if phone number is already registered permanently
    $existing_phone = $wpdb->get_var($wpdb->prepare(
        "SELECT company_phone FROM {$wpdb->prefix}company_registration WHERE company_phone = %s AND temporary = 0",
        $phone
    ));

    if ($existing_phone) {
        return new WP_REST_Response(['success' => false, 'message' => 'Company phone number is already registered.'], 400);
    }

    // Generate a new verification code
    $verification_code = wp_generate_password(6, false);

    // Check if a temporary record exists for the email
    $existing_temp_user = $wpdb->get_row($wpdb->prepare(
        "SELECT id FROM {$wpdb->prefix}company_registration WHERE company_email = %s AND temporary = 1",
        $email
    ));

    if ($existing_temp_user) {
        // Update the existing temporary record with the new verification code
        $wpdb->update(
            $wpdb->prefix . 'company_registration',
            ['verification_code' => $verification_code],
            ['id' => $existing_temp_user->id]
        );
    } else {
        // Create a new temporary record
        $wpdb->insert(
            $wpdb->prefix . 'company_registration',
            [
                'company_name' => '', // Optional, since it's a resend
                'company_email' => $email,
                'company_phone' => $phone,
                'verification_code' => $verification_code,
                'verified' => 0,
                'temporary' => 1, // Mark as temporary
            ]
        );
    }

    // Send the verification code via email
    $subject = "Your Instock Verification Code";
    $message = "Your Instock verification code is: $verification_code";

    if (wp_mail($email, $subject, $message)) {
        return new WP_REST_Response(['success' => true, 'message' => 'Verification code sent successfully.'], 200);
    } else {
        return new WP_REST_Response(['success' => false, 'message' => 'Failed to send the verification code.'], 500);
    }
}


// Function to post data to the external endpoint
function post_data_to_external_endpoint($company_name, $company_email, $company_phone, $password) {
    $url = 'https://instock.lk/save-register';

    $data = [
        'company_name' => $company_name,
        'company_email' => $company_email,
        'company_phone' => $company_phone,
        'password' => $password
    ];

    $response = wp_remote_post($url, [
        'method' => 'POST',
        'body' => $data,
        'timeout' => 45,
        'headers' => ['Content-Type' => 'application/x-www-form-urlencoded']
    ]);

    if (is_wp_error($response)) {
        $error_message = $response->get_error_message();
        error_log("Error posting data to external endpoint: $error_message");
        return false;
    }

    $response_code = wp_remote_retrieve_response_code($response);
    return $response_code === 200;
}

// Function to handle registration
function save_registration_data() {
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action']) && $_GET['action'] === 'save-user') {
        global $wpdb;

        $company_name = sanitize_text_field($_POST['company_name']);
        $company_email = sanitize_email($_POST['company_email']);
        $company_phone = sanitize_text_field($_POST['company_phone']);
        $password = sanitize_text_field($_POST['password']);
        $confirm_password = sanitize_text_field($_POST['confirm_password']);
        $verification_code = sanitize_text_field($_POST['verification_code']);

        // Check if verification code matches
        $user = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}company_registration WHERE company_email = %s AND verification_code = %s AND temporary = 1",
            $company_email,
            $verification_code
        ));

        if (!$user) {
            wp_send_json_error(['message' => 'Invalid verification code.']);
            return;
        }

        // Check if passwords match
        if ($password !== $confirm_password) {
            wp_send_json_error(['message' => 'Passwords do not match.']);
            return;
        }

        // Update user record as permanent
        $wpdb->update(
            $wpdb->prefix . 'company_registration',
            [
                'company_name' => $company_name,
                'company_phone' => $company_phone,
                'password' => wp_hash_password($password),
                'verified' => 1,
                'temporary' => 0, // Mark as permanent
                'verification_code' => null,
            ],
            ['company_email' => $company_email]
        );

        // Post data to external API
        $external_api_success = post_data_to_external_endpoint($company_name, $company_email, $company_phone, $password);

        if ($external_api_success) {
            // Create WordPress user as a subscriber
            $user_id = wp_create_user($company_email, $password, $company_email);
            if (!is_wp_error($user_id)) {
                $wp_user = new WP_User($user_id);
                $wp_user->set_role('subscriber');
            }
            wp_send_json_success(['message' => 'Registration successful! Redirecting to login…']);
        } else {
            wp_send_json_error(['message' => 'Error posting data to external endpoint.']);
        }
    }
}
add_action('init', 'save_registration_data');

// Change the sender email address
add_filter('wp_mail_from', function ($original_email_address) {
    return 'noreply@instock.lk'; // Replace with your desired email address
});

// Change the sender name
add_filter('wp_mail_from_name', function ($original_email_from) {
    return 'Instock'; // Replace with your desired sender name
});
